const mongoose = require('mongoose');
const School = require('../models/School');

// Connect to database
const connectDB = require('../utils/db');
connectDB();

async function checkSchoolData() {
    try {
        console.log('🔍 [DATA CHECK] Checking school data in database...\n');
        
        // Check total documents
        const totalSchools = await School.countDocuments();
        console.log(`📊 [DATA CHECK] Total schools in database: ${totalSchools}`);
        
        if (totalSchools === 0) {
            console.log('❌ [DATA CHECK] No schools found in database!');
            return;
        }
        
        // Check unique village codes
        const villageCodes = await School.distinct('udise_village_code');
        console.log(`📊 [DATA CHECK] Unique village codes: ${villageCodes.length}`);
        console.log(`📊 [DATA CHECK] First 10 village codes:`, villageCodes.slice(0, 10));
        
        // Check unique district codes
        const districtCodes = await School.distinct('district_code');
        console.log(`📊 [DATA CHECK] Unique district codes: ${districtCodes.length}`);
        console.log(`📊 [DATA CHECK] First 10 district codes:`, districtCodes.slice(0, 10));
        
        // Check unique state codes
        const stateCodes = await School.distinct('state_code');
        console.log(`📊 [DATA CHECK] Unique state codes: ${stateCodes.length}`);
        console.log(`📊 [DATA CHECK] First 10 state codes:`, stateCodes.slice(0, 10));
        
        // Get sample data
        console.log('\n📋 [DATA CHECK] Sample school records:');
        const sampleSchools = await School.find().limit(5).select('state_name state_code district_name district_code village_name udise_village_code school_name udise_school_code');
        sampleSchools.forEach((school, index) => {
            console.log(`\n${index + 1}. School: ${school.school_name}`);
            console.log(`   State: ${school.state_name} (${school.state_code})`);
            console.log(`   District: ${school.district_name} (${school.district_code})`);
            console.log(`   Village: ${school.village_name} (${school.udise_village_code})`);
            console.log(`   School Code: ${school.udise_school_code}`);
        });
        
        // Test a specific village code
        if (villageCodes.length > 0) {
            const testVillageCode = villageCodes[0];
            console.log(`\n🧪 [DATA CHECK] Testing village code: ${testVillageCode}`);
            
            const schoolsForVillage = await School.find({ udise_village_code: testVillageCode });
            console.log(`📊 [DATA CHECK] Schools for village ${testVillageCode}: ${schoolsForVillage.length}`);
            
            if (schoolsForVillage.length > 0) {
                console.log(`✅ [DATA CHECK] Found schools for village ${testVillageCode}:`);
                schoolsForVillage.forEach((school, index) => {
                    console.log(`  ${index + 1}. ${school.school_name} (${school.udise_school_code})`);
                });
            } else {
                console.log(`⚠️ [DATA CHECK] No schools found for village ${testVillageCode}`);
            }
        }
        
        // Check data types
        console.log('\n🔍 [DATA CHECK] Checking data types...');
        const sampleSchool = await School.findOne();
        if (sampleSchool) {
            console.log(`📊 [DATA CHECK] Sample data types:`);
            console.log(`   state_code: ${typeof sampleSchool.state_code} (${sampleSchool.state_code})`);
            console.log(`   district_code: ${typeof sampleSchool.district_code} (${sampleSchool.district_code})`);
            console.log(`   udise_village_code: ${typeof sampleSchool.udise_village_code} (${sampleSchool.udise_village_code})`);
            console.log(`   udise_school_code: ${typeof sampleSchool.udise_school_code} (${sampleSchool.udise_school_code})`);
        }
        
    } catch (error) {
        console.error('❌ [DATA CHECK ERROR] Failed to check school data:', error);
    } finally {
        // Close database connection
        mongoose.connection.close();
    }
}

// Run the check
checkSchoolData();
