const express = require('express');
const router = express.Router();
const portfolioController = require('../controllers/portfolioController');
const { authenticateToken } = require('../middleware/auth');
const multer = require('multer');

// Configure multer for file uploads
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, 'uploads/portfolio/');
    },
    filename: function (req, file, cb) {
        cb(null, Date.now() + '-' + file.originalname);
    }
});

const upload = multer({ 
    storage: storage,
    limits: {
        fileSize: 10 * 1024 * 1024 // 10MB limit
    },
    fileFilter: function (req, file, cb) {
        // Allow common document types
        const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|txt/;
        const extname = allowedTypes.test(file.originalname.toLowerCase());
        const mimetype = allowedTypes.test(file.mimetype);

        if (mimetype && extname) {
            return cb(null, true);
        } else {
            cb(new Error('Only documents and images are allowed'));
        }
    }
});

// Portfolio Routes (Protected)
router.post('/', authenticateToken, portfolioController.createOrUpdatePortfolio);
router.get('/', authenticateToken, portfolioController.getUserPortfolio);
router.put('/visibility', authenticateToken, portfolioController.togglePortfolioVisibility);

// Portfolio Items (Protected)
router.post('/education', authenticateToken, portfolioController.addEducation);
router.post('/skills', authenticateToken, portfolioController.addSkill);
router.post('/projects', authenticateToken, portfolioController.addProject);
router.post('/work-experience', authenticateToken, portfolioController.addWorkExperience);
router.post('/documents', authenticateToken, upload.single('document'), portfolioController.uploadDocument);

// Delete Items (Protected)
router.delete('/:section/:itemId', authenticateToken, portfolioController.deletePortfolioItem);

// Public Routes
router.get('/public/:userId', portfolioController.getPublicPortfolio);

module.exports = router;
