const CareerPath = require('../models/CareerPath');

// Get all career paths (Admin only)
const getAllCareerPaths = async (req, res) => {
    try {
        const { page = 1, limit = 50, stage, difficulty, demand, isActive } = req.query;

        const filter = {};

        if (stage) filter.stage = stage;
        if (difficulty) filter.difficulty = difficulty;
        if (demand) filter.demand = demand;
        if (isActive !== undefined) filter.isActive = isActive === 'true';

        const careerPaths = await CareerPath.find(filter)
            .select('-__v')
            .sort({ createdAt: -1 })
            .limit(limit * 1)
            .skip((page - 1) * limit);

        const total = await CareerPath.countDocuments(filter);

        res.status(200).json({
            success: true,
            data: {
                careerPaths,
                pagination: {
                    currentPage: parseInt(page),
                    totalPages: Math.ceil(total / limit),
                    totalPaths: total,
                    hasNext: page < Math.ceil(total / limit),
                    hasPrev: page > 1
                }
            }
        });
    } catch (error) {
        console.error('Get all career paths error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get career paths',
            error: error.message
        });
    }
};

// Get career path details
const getCareerPathDetails = async (req, res) => {
    try {
        const { pathId } = req.params;
        const careerPath = await CareerPath.findById(pathId);
        
        if (!careerPath) {
            return res.status(404).json({
                success: false,
                message: 'Career path not found'
            });
        }
        
        res.status(200).json({
            success: true,
            data: careerPath
        });
    } catch (error) {
        console.error('Get career path details error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get career path details',
            error: error.message
        });
    }
};

// Add new career path (Admin only)
const addCareerPath = async (req, res) => {
    try {
        const careerPathData = req.body;

        const careerPath = new CareerPath(careerPathData);
        await careerPath.save();

        res.status(201).json({
            success: true,
            message: 'Career path added successfully',
            data: careerPath
        });
    } catch (error) {
        console.error('Add career path error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to add career path',
            error: error.message
        });
    }
};

// Update career path (Admin only)
const updateCareerPath = async (req, res) => {
    try {
        const { pathId } = req.params;
        const updateData = req.body;

        const careerPath = await CareerPath.findByIdAndUpdate(
            pathId,
            updateData,
            { new: true, runValidators: true }
        );

        if (!careerPath) {
            return res.status(404).json({
                success: false,
                message: 'Career path not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Career path updated successfully',
            data: careerPath
        });
    } catch (error) {
        console.error('Update career path error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to update career path',
            error: error.message
        });
    }
};

// Delete career path (Admin only)
const deleteCareerPath = async (req, res) => {
    try {
        const { pathId } = req.params;

        const careerPath = await CareerPath.findByIdAndDelete(pathId);

        if (!careerPath) {
            return res.status(404).json({
                success: false,
                message: 'Career path not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Career path deleted successfully'
        });
    } catch (error) {
        console.error('Delete career path error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to delete career path',
            error: error.message
        });
    }
};

module.exports = {
    getAllCareerPaths,
    getCareerPathDetails,
    addCareerPath,
    updateCareerPath,
    deleteCareerPath
};
