const axios = require('axios');

const BASE_URL = 'http://localhost:5500/api/news';

// Test functions
const testGetNews = async () => {
    try {
        console.log('🧪 Testing GET /api/news...');
        const response = await axios.get(`${BASE_URL}`);
        console.log('✅ Get news - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            totalArticles: response.data.data?.pagination?.total || 0,
            articlesCount: response.data.data?.news?.length || 0
        });
    } catch (error) {
        console.error('❌ Get news error:', error.response?.data || error.message);
    }
};

const testGetNewsByCategory = async () => {
    try {
        console.log('🧪 Testing GET /api/news/category/technology...');
        const response = await axios.get(`${BASE_URL}/category/technology`);
        console.log('✅ Get news by category - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            totalArticles: response.data.data?.pagination?.total || 0,
            articlesCount: response.data.data?.news?.length || 0
        });
    } catch (error) {
        console.error('❌ Get news by category error:', error.response?.data || error.message);
    }
};

const testSearchNews = async () => {
    try {
        console.log('🧪 Testing GET /api/news/search?q=technology...');
        const response = await axios.get(`${BASE_URL}/search?q=technology`);
        console.log('✅ Search news - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            totalArticles: response.data.data?.pagination?.total || 0,
            articlesCount: response.data.data?.news?.length || 0
        });
    } catch (error) {
        console.error('❌ Search news error:', error.response?.data || error.message);
    }
};

const testGetCategories = async () => {
    try {
        console.log('🧪 Testing GET /api/news/categories...');
        const response = await axios.get(`${BASE_URL}/categories`);
        console.log('✅ Get categories - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            categoriesCount: response.data.data?.length || 0
        });
    } catch (error) {
        console.error('❌ Get categories error:', error.response?.data || error.message);
    }
};

const testGetSources = async () => {
    try {
        console.log('🧪 Testing GET /api/news/sources...');
        const response = await axios.get(`${BASE_URL}/sources`);
        console.log('✅ Get sources - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            sourcesCount: response.data.data?.length || 0
        });
    } catch (error) {
        console.error('❌ Get sources error:', error.response?.data || error.message);
    }
};

const testGetLatestNews = async () => {
    try {
        console.log('🧪 Testing GET /api/news/latest...');
        const response = await axios.get(`${BASE_URL}/latest`);
        console.log('✅ Get latest news - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            articlesCount: response.data.data?.length || 0
        });
    } catch (error) {
        console.error('❌ Get latest news error:', error.response?.data || error.message);
    }
};

const testGetTrendingNews = async () => {
    try {
        console.log('🧪 Testing GET /api/news/trending...');
        const response = await axios.get(`${BASE_URL}/trending`);
        console.log('✅ Get trending news - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            articlesCount: response.data.data?.length || 0
        });
    } catch (error) {
        console.error('❌ Get trending news error:', error.response?.data || error.message);
    }
};

const testGetNewsStats = async () => {
    try {
        console.log('🧪 Testing GET /api/news/stats...');
        const response = await axios.get(`${BASE_URL}/stats`);
        console.log('✅ Get news stats - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            stats: response.data.data
        });
    } catch (error) {
        console.error('❌ Get news stats error:', error.response?.data || error.message);
    }
};

const testRefreshNews = async () => {
    try {
        console.log('🧪 Testing POST /api/news/refresh...');
        const response = await axios.post(`${BASE_URL}/refresh`, {
            categories: ['technology', 'business']
        });
        console.log('✅ Refresh news - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            message: response.data.message,
            result: response.data.data
        });
    } catch (error) {
        console.error('❌ Refresh news error:', error.response?.data || error.message);
    }
};

const testClearCache = async () => {
    try {
        console.log('🧪 Testing POST /api/news/cache/clear...');
        const response = await axios.post(`${BASE_URL}/cache/clear`);
        console.log('✅ Clear cache - Status:', response.status);
        console.log('📊 Response data:', {
            success: response.data.success,
            message: response.data.message
        });
    } catch (error) {
        console.error('❌ Clear cache error:', error.response?.data || error.message);
    }
};

// Run all tests
const runAllTests = async () => {
    console.log('🚀 Starting News API Tests...\n');
    
    await testGetNews();
    console.log('');
    
    await testGetNewsByCategory();
    console.log('');
    
    await testSearchNews();
    console.log('');
    
    await testGetCategories();
    console.log('');
    
    await testGetSources();
    console.log('');
    
    await testGetLatestNews();
    console.log('');
    
    await testGetTrendingNews();
    console.log('');
    
    await testGetNewsStats();
    console.log('');
    
    await testClearCache();
    console.log('');
    
    // Uncomment to test refresh (this will make API calls)
    // await testRefreshNews();
    // console.log('');
    
    console.log('🎉 All tests completed!');
};

// Run tests if this file is executed directly
if (require.main === module) {
    runAllTests().catch(console.error);
}

module.exports = {
    testGetNews,
    testGetNewsByCategory,
    testSearchNews,
    testGetCategories,
    testGetSources,
    testGetLatestNews,
    testGetTrendingNews,
    testGetNewsStats,
    testRefreshNews,
    testClearCache,
    runAllTests
};
