const { body, validationResult } = require('express-validator');

// Input validation middleware for registration
const validateRegistration = [
   
    
    body('dob')
        .isISO8601()
        .withMessage('Date of birth must be a valid date')
        .custom((value) => {
            const dob = new Date(value);
            const today = new Date();
            const age = today.getFullYear() - dob.getFullYear();
            
            if (age < 13) {
                throw new Error('User must be at least 13 years old');
            }
            if (age > 120) {
                throw new Error('Invalid date of birth');
            }
            return true;
        }),
    
    body('mobile')
        .trim()
        .isLength({ min: 10, max: 15 })
        .withMessage('Mobile number must be between 10 and 15 characters'),
    
    body('password')
        .isLength({ min: 6, max: 128 })
        .withMessage('Password must be between 6 and 128 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for login
const validateLogin = [
    body('mobile')
        .trim()
        .isLength({ min: 10, max: 15 })
        .withMessage('Please provide a valid mobile number'),
    
    body('password')
        .notEmpty()
        .withMessage('Password is required')
        .isLength({ min: 1, max: 128 })
        .withMessage('Password must not exceed 128 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for OTP verification
const validateOTP = [
    body('mobile')
        .trim()
        .isMobilePhone('any')
        .withMessage('Please provide a valid mobile number'),
    
    body('otp')
        .trim()
        .isLength({ min: 6, max: 6 })
        .withMessage('OTP must be exactly 6 digits')
        .isNumeric()
        .withMessage('OTP must contain only numbers'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for personal details
const validatePersonalDetails = [
    body('fullName')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Full name must be between 2 and 100 characters')
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Full name can only contain letters and spaces'),
    
    body('gender')
        .isIn(['Male', 'Female', 'Other'])
        .withMessage('Gender must be Male, Female, or Other'),
    
    body('emailId')
        .trim()
        .isEmail()
        .withMessage('Please provide a valid email address')
        .normalizeEmail(),
    
    body('bloodGroup')
        .isIn(['A+', 'A-', 'B+', 'B-', 'AB+', 'AB-', 'O+', 'O-'])
        .withMessage('Blood group must be one of the valid types'),
    
    body('pincodeNative')
        .trim()
        .matches(/^[1-9][0-9]{5}$/)
        .withMessage('Pincode must be a valid 6-digit number'),
    
    body('state')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('State must be between 2 and 50 characters'),
    
    body('district')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('District must be between 2 and 50 characters'),
    
    body('nativePlace')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Native place must be between 2 and 100 characters'),
    
    
    body('nameMeaning')
        .optional()
        .trim()
        .isLength({ max: 500 })
        .withMessage('Name meaning cannot exceed 500 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Partial validation for updating personal details
const validatePersonalDetailsUpdate = [
    body('fullName')
        .optional()
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Full name must be between 2 and 100 characters')
        .matches(/^[a-zA-Z\s]+$/)
        .withMessage('Full name can only contain letters and spaces'),
    
    body('gender')
        .optional()
        .isIn(['Male', 'Female', 'Other'])
        .withMessage('Gender must be Male, Female, or Other'),
    
    body('emailId')
        .optional()
        .trim()
        .isEmail()
        .withMessage('Please provide a valid email address')
        .normalizeEmail(),
    
    body('bloodGroup')
        .optional()
        .isIn(['A+', 'A-', 'B+', 'B-', 'AB+', 'AB-', 'O+', 'O-'])
        .withMessage('Blood group must be one of the valid types'),
    
    body('pincodeNative')
        .optional()
        .trim()
        .matches(/^[1-9][0-9]{5}$/)
        .withMessage('Pincode must be a valid 6-digit number'),
    
    body('state')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('State must be between 2 and 50 characters'),
    
    body('district')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('District must be between 2 and 50 characters'),
    
    body('nativePlace')
        .optional()
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Native place must be between 2 and 100 characters'),
    
    
    body('nameMeaning')
        .optional()
        .trim()
        .isLength({ max: 500 })
        .withMessage('Name meaning cannot exceed 500 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];


// Input validation middleware for school details
const validateSchoolDetails = [
    body('currentClass')
        .isIn(['6th', '7th', '8th', '9th', '10th'])
        .withMessage('Current class must be one of: 6th, 7th, 8th, 9th, 10th'),
    
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of: SSC/State Board, CBSE, ICSE, IB, Cambridge, Other'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of: Co-ed, Boys School, Girls School, Other'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    body('sectionOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other section cannot exceed 100 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Partial validation for updating school details
const validateSchoolDetailsUpdate = [
    body('currentClass')
        .optional()
        .isIn(['6th', '7th', '8th', '9th', '10th'])
        .withMessage('Current class must be one of: 6th, 7th, 8th, 9th, 10th'),
    
    
    body('schoolState')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .optional()
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .optional()
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of: SSC/State Board, CBSE, ICSE, IB, Cambridge, Other'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .optional()
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of: Co-ed, Boys School, Girls School, Other'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    body('sectionOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other section cannot exceed 100 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for graduate details
const validateGraduateDetails = [
    // School Details (from Educational Details)
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('yearOfJoiningSchool')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    // College Details
    
    body('collegeState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College state must be between 2 and 50 characters'),
    
    body('collegeDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College district must be between 2 and 50 characters'),
    
    body('collegeName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('College name must be between 2 and 200 characters'),
    
    body('graduationJoiningYear')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Graduation joining year must be between 1900 and current year'),
    
    body('mainStream')
        .isIn(['BA', 'BCom', 'BTech', 'BSc', 'BBA', 'BCA', 'BPharm', 'BDS', 'MBBS', 'BArch', 'BFA', 'BHM', 'LLB', 'Other'])
        .withMessage('Main stream must be one of the predefined options'),
    
    body('mainStreamOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other main stream cannot exceed 100 characters'),
    
    body('branchStream')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Branch/Stream must be between 2 and 100 characters'),
    
    body('collegeType')
        .isIn(['Co-ed', 'Boys College', 'Girls College', 'Other'])
        .withMessage('College type must be one of: Co-ed, Boys College, Girls College, Other'),
    
    body('collegeTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other college type cannot exceed 100 characters'),
    
    // PG Details
    body('isPursuingOrCompletedPG')
        .optional()
        .isBoolean()
        .withMessage('Is pursuing/completed PG must be a boolean'),
    
    body('pgCollegeName')
        .optional()
        .trim()
        .isLength({ max: 200 })
        .withMessage('PG college name cannot exceed 200 characters'),
    
    body('pgJoiningYear')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('PG joining year must be between 1900 and current year'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Partial validation for updating graduate details
const validateGraduateDetailsUpdate = [
    // School Details (from Educational Details)
    
    body('schoolState')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .optional()
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('yearOfJoiningSchool')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    // College Details
    
    body('collegeState')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College state must be between 2 and 50 characters'),
    
    body('collegeDistrict')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College district must be between 2 and 50 characters'),
    
    body('collegeName')
        .optional()
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('College name must be between 2 and 200 characters'),
    
    body('graduationJoiningYear')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Graduation joining year must be between 1900 and current year'),
    
    body('mainStream')
        .optional()
        .isIn(['BA', 'BCom', 'BTech', 'BSc', 'BBA', 'BCA', 'BPharm', 'BDS', 'MBBS', 'BArch', 'BFA', 'BHM', 'LLB', 'Other'])
        .withMessage('Main stream must be one of the predefined options'),
    
    body('mainStreamOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other main stream cannot exceed 100 characters'),
    
    body('branchStream')
        .optional()
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Branch/Stream must be between 2 and 100 characters'),
    
    body('collegeType')
        .optional()
        .isIn(['Co-ed', 'Boys College', 'Girls College', 'Other'])
        .withMessage('College type must be one of: Co-ed, Boys College, Girls College, Other'),
    
    body('collegeTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other college type cannot exceed 100 characters'),
    
    // PG Details
    body('isPursuingOrCompletedPG')
        .optional()
        .isBoolean()
        .withMessage('Is pursuing/completed PG must be a boolean'),
    
    body('pgCollegeName')
        .optional()
        .trim()
        .isLength({ max: 200 })
        .withMessage('PG college name cannot exceed 200 characters'),
    
    body('pgJoiningYear')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('PG joining year must be between 1900 and current year'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for adding schools to database
const validateSchoolAddition = [
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolAddress')
        .optional()
        .trim()
        .isLength({ max: 500 })
        .withMessage('School address cannot exceed 500 characters'),
    
    body('boardType')
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of: SSC/State Board, CBSE, ICSE, IB, Cambridge, Other'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of: Co-ed, Boys School, Girls School, Other'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('availableClasses')
        .optional()
        .isArray()
        .withMessage('Available classes must be an array'),
    
    body('availableClasses.*')
        .optional()
        .isIn(['6th', '7th', '8th', '9th', '10th'])
        .withMessage('Each class must be one of: 6th, 7th, 8th, 9th, 10th'),
    
    body('schoolWebsite')
        .optional()
        .isURL()
        .withMessage('School website must be a valid URL'),
    
    body('schoolPhone')
        .optional()
        .trim()
        .isLength({ max: 15 })
        .withMessage('School phone cannot exceed 15 characters'),
    
    body('schoolEmail')
        .optional()
        .isEmail()
        .withMessage('School email must be a valid email address'),
    
    body('udiseCode')
        .optional()
        .trim()
        .isLength({ max: 20 })
        .withMessage('UDISE code cannot exceed 20 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for adding colleges to database
const validateCollegeAddition = [
    body('collegeName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('College name must be between 2 and 200 characters'),
    
    
    body('collegeState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College state must be between 2 and 50 characters'),
    
    body('collegeDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College district must be between 2 and 50 characters'),
    
    body('collegeAddress')
        .optional()
        .trim()
        .isLength({ max: 500 })
        .withMessage('College address cannot exceed 500 characters'),
    
    body('mainStream')
        .isIn(['BA', 'BCom', 'BTech', 'BSc', 'BBA', 'BCA', 'BPharm', 'BDS', 'MBBS', 'BArch', 'BFA', 'BHM', 'LLB', 'Other'])
        .withMessage('Main stream must be one of the predefined options'),
    
    body('mainStreamOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other main stream cannot exceed 100 characters'),
    
    body('availableBranches')
        .optional()
        .isArray()
        .withMessage('Available branches must be an array'),
    
    body('availableBranches.*')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Each branch name cannot exceed 100 characters'),
    
    body('collegeType')
        .isIn(['Co-ed', 'Boys College', 'Girls College', 'Other'])
        .withMessage('College type must be one of: Co-ed, Boys College, Girls College, Other'),
    
    body('collegeTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other college type cannot exceed 100 characters'),
    
    body('collegeWebsite')
        .optional()
        .isURL()
        .withMessage('College website must be a valid URL'),
    
    body('collegePhone')
        .optional()
        .trim()
        .isLength({ max: 15 })
        .withMessage('College phone cannot exceed 15 characters'),
    
    body('collegeEmail')
        .optional()
        .isEmail()
        .withMessage('College email must be a valid email address'),
    
    body('collegeCode')
        .optional()
        .trim()
        .isLength({ max: 20 })
        .withMessage('College code cannot exceed 20 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for working professional details
const validateWorkingProfessionalDetails = [
    // Job Entries validation
    body('jobEntries')
        .isArray({ min: 1 })
        .withMessage('At least one job entry is required'),
    
    body('jobEntries.*.jobRole')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Job role must be between 2 and 100 characters'),
    
    body('jobEntries.*.companyName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('Company name must be between 2 and 200 characters'),
    
    body('jobEntries.*.industry')
        .isIn(['IT', 'Government', 'Finance', 'Healthcare', 'Education', 'Manufacturing', 'Retail', 'Real Estate', 'Media', 'Consulting', 'Other'])
        .withMessage('Industry must be one of the predefined options'),
    
    body('jobEntries.*.industryOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other industry cannot exceed 100 characters'),
    
    body('jobEntries.*.jobPincode')
        .matches(/^[1-9][0-9]{5}$/)
        .withMessage('Job pincode must be a valid 6-digit number'),
    
    body('jobEntries.*.jobState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Job state must be between 2 and 50 characters'),
    
    body('jobEntries.*.jobDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Job district must be between 2 and 50 characters'),
    
    body('jobEntries.*.jobCity')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Job city must be between 2 and 100 characters'),
    
    body('jobEntries.*.jobLocality')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('Job locality must be between 2 and 200 characters'),
    
    body('jobEntries.*.workplaceType')
        .optional()
        .isIn(['On-site', 'Remote', 'Hybrid'])
        .withMessage('Workplace type must be one of: On-site, Remote, Hybrid'),
    
    body('jobEntries.*.startDate')
        .optional()
        .isISO8601()
        .withMessage('Start date must be a valid date'),
    
    body('jobEntries.*.endDate')
        .optional()
        .isISO8601()
        .withMessage('End date must be a valid date'),
    
    body('jobEntries.*.isCurrentJob')
        .optional()
        .isBoolean()
        .withMessage('Is current job must be a boolean'),
    
    // School Details validation
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of the predefined options'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of the predefined options'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    // Graduate Details validation
    
    body('collegeState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College state must be between 2 and 50 characters'),
    
    body('collegeDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College district must be between 2 and 50 characters'),
    
    body('collegeName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('College name must be between 2 and 200 characters'),
    
    body('graduationJoiningYear')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Graduation joining year must be between 1900 and current year'),
    
    body('mainStream')
        .isIn(['BA', 'BCom', 'BTech', 'BSc', 'BBA', 'BCA', 'BPharm', 'BDS', 'MBBS', 'BArch', 'BFA', 'BHM', 'LLB', 'Other'])
        .withMessage('Main stream must be one of the predefined options'),
    
    body('mainStreamOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other main stream cannot exceed 100 characters'),
    
    body('branchStream')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Branch/Stream must be between 2 and 100 characters'),
    
    body('collegeType')
        .isIn(['Co-ed', 'Boys College', 'Girls College', 'Other'])
        .withMessage('College type must be one of the predefined options'),
    
    body('collegeTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other college type cannot exceed 100 characters'),
    
    // PG Details validation
    body('isPursuingOrCompletedPG')
        .optional()
        .isBoolean()
        .withMessage('Is pursuing or completed PG must be a boolean'),
    
    body('pgCollegeName')
        .optional()
        .trim()
        .isLength({ max: 200 })
        .withMessage('PG college name cannot exceed 200 characters'),
    
    body('pgJoiningYear')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('PG joining year must be between 1900 and current year'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for single job entry
const validateJobEntry = [
    body('jobRole')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Job role must be between 2 and 100 characters'),
    
    body('companyName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('Company name must be between 2 and 200 characters'),
    
    body('industry')
        .isIn(['IT', 'Government', 'Finance', 'Healthcare', 'Education', 'Manufacturing', 'Retail', 'Real Estate', 'Media', 'Consulting', 'Other'])
        .withMessage('Industry must be one of the predefined options'),
    
    body('industryOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other industry cannot exceed 100 characters'),
    
    body('jobPincode')
        .matches(/^[1-9][0-9]{5}$/)
        .withMessage('Job pincode must be a valid 6-digit number'),
    
    body('jobState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Job state must be between 2 and 50 characters'),
    
    body('jobDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Job district must be between 2 and 50 characters'),
    
    body('jobCity')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Job city must be between 2 and 100 characters'),
    
    body('jobLocality')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('Job locality must be between 2 and 200 characters'),
    
    body('workplaceType')
        .optional()
        .isIn(['On-site', 'Remote', 'Hybrid'])
        .withMessage('Workplace type must be one of: On-site, Remote, Hybrid'),
    
    body('startDate')
        .optional()
        .isISO8601()
        .withMessage('Start date must be a valid date'),
    
    body('endDate')
        .optional()
        .isISO8601()
        .withMessage('End date must be a valid date'),
    
    body('isCurrentJob')
        .optional()
        .isBoolean()
        .withMessage('Is current job must be a boolean'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for business details
const validateBusinessDetails = [
    // School Details validation
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of the predefined options'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of the predefined options'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    // Graduate Details validation
    
    body('collegeState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College state must be between 2 and 50 characters'),
    
    body('collegeDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('College district must be between 2 and 50 characters'),
    
    body('collegeName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('College name must be between 2 and 200 characters'),
    
    body('graduationJoiningYear')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Graduation joining year must be between 1900 and current year'),
    
    body('mainStream')
        .isIn(['BA', 'BCom', 'BTech', 'BSc', 'BBA', 'BCA', 'BPharm', 'BDS', 'MBBS', 'BArch', 'BFA', 'BHM', 'LLB', 'Other'])
        .withMessage('Main stream must be one of the predefined options'),
    
    body('mainStreamOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other main stream cannot exceed 100 characters'),
    
    body('branchStream')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Branch/Stream must be between 2 and 100 characters'),
    
    body('collegeType')
        .isIn(['Co-ed', 'Boys College', 'Girls College', 'Other'])
        .withMessage('College type must be one of the predefined options'),
    
    body('collegeTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other college type cannot exceed 100 characters'),
    
    // PG Details validation
    body('isPursuingOrCompletedPG')
        .optional()
        .isBoolean()
        .withMessage('Is pursuing or completed PG must be a boolean'),
    
    body('pgCollegeName')
        .optional()
        .trim()
        .isLength({ max: 200 })
        .withMessage('PG college name cannot exceed 200 characters'),
    
    body('pgJoiningYear')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('PG joining year must be between 1900 and current year'),
    
    // Business Details validation
    body('businessName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('Business name must be between 2 and 200 characters'),
    
    body('businessType')
        .isIn(['Startup', 'MSME', 'Freelancer', 'Pvt Ltd'])
        .withMessage('Business type must be one of: Startup, MSME, Freelancer, Pvt Ltd'),
    
    body('industry')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Industry must be between 2 and 100 characters'),
    
    body('yearsRunning')
        .isIn(['0-1', '1-3', '3-5', '5+'])
        .withMessage('Years running must be one of: 0-1, 1-3, 3-5, 5+'),
    
    // Business Location validation
    body('businessPincode')
        .matches(/^[1-9][0-9]{5}$/)
        .withMessage('Business pincode must be a valid 6-digit number'),
    
    body('businessState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Business state must be between 2 and 50 characters'),
    
    body('businessDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('Business district must be between 2 and 50 characters'),
    
    body('businessTown')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Business town must be between 2 and 100 characters'),
    
    body('websiteOrInstaLink')
        .optional()
        .trim()
        .isLength({ max: 200 })
        .withMessage('Website/Instagram link cannot exceed 200 characters'),
    
    // Business POC validation
    body('businessPOCName')
        .trim()
        .isLength({ min: 2, max: 100 })
        .withMessage('Business POC name must be between 2 and 100 characters'),
    
    body('businessPOCMobile')
        .matches(/^[6-9]\d{9}$/)
        .withMessage('Please enter a valid 10-digit mobile number'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for post creation
const validatePostCreation = [
    body('title')
        .trim()
        .isLength({ min: 1, max: 200 })
        .withMessage('Title must be between 1 and 200 characters'),
    
    body('description')
        .trim()
        .isLength({ min: 1, max: 2000 })
        .withMessage('Description must be between 1 and 2000 characters'),
    
    body('tags')
        .optional()
        .isString()
        .withMessage('Tags must be a string'),
    
    body('category')
        .optional()
        .isIn(['general', 'announcement', 'event', 'news', 'promotion', 'educational'])
        .withMessage('Category must be one of: general, announcement, event, news, promotion, educational'),
    
    body('visibility')
        .optional()
        .isIn(['public', 'private', 'friends_only'])
        .withMessage('Visibility must be one of: public, private, friends_only'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for post update
const validatePostUpdate = [
    body('title')
        .optional()
        .trim()
        .isLength({ min: 1, max: 200 })
        .withMessage('Title must be between 1 and 200 characters'),
    
    body('description')
        .optional()
        .trim()
        .isLength({ min: 1, max: 2000 })
        .withMessage('Description must be between 1 and 2000 characters'),
    
    body('tags')
        .optional()
        .isString()
        .withMessage('Tags must be a string'),
    
    body('category')
        .optional()
        .isIn(['general', 'announcement', 'event', 'news', 'promotion', 'educational'])
        .withMessage('Category must be one of: general, announcement, event, news, promotion, educational'),
    
    body('visibility')
        .optional()
        .isIn(['public', 'private', 'friends_only'])
        .withMessage('Visibility must be one of: public, private, friends_only'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for comment
const validateComment = [
    body('content')
        .trim()
        .isLength({ min: 1, max: 500 })
        .withMessage('Comment content must be between 1 and 500 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Input validation middleware for school details without current class
const validateSchoolDetailsWithoutCurrentClass = [
    
    body('schoolState')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolVillage')
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School village must be between 2 and 50 characters'),
    
    body('schoolName')
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of: SSC/State Board, CBSE, ICSE, IB, Cambridge, Other'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of: Co-ed, Boys School, Girls School, Other'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

// Partial validation for updating school details without current class
const validateSchoolDetailsWithoutCurrentClassUpdate = [
    
    body('schoolState')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School state must be between 2 and 50 characters'),
    
    body('schoolDistrict')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School district must be between 2 and 50 characters'),
    
    body('schoolVillage')
        .optional()
        .trim()
        .isLength({ min: 2, max: 50 })
        .withMessage('School village must be between 2 and 50 characters'),
    
    body('schoolName')
        .optional()
        .trim()
        .isLength({ min: 2, max: 200 })
        .withMessage('School name must be between 2 and 200 characters'),
    
    body('boardType')
        .optional()
        .isIn(['SSC/State Board', 'CBSE', 'ICSE', 'IB', 'Cambridge', 'Other'])
        .withMessage('Board type must be one of: SSC/State Board, CBSE, ICSE, IB, Cambridge, Other'),
    
    body('boardTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other board type cannot exceed 100 characters'),
    
    body('schoolType')
        .optional()
        .isIn(['Co-ed', 'Boys School', 'Girls School', 'Other'])
        .withMessage('School type must be one of: Co-ed, Boys School, Girls School, Other'),
    
    body('schoolTypeOther')
        .optional()
        .trim()
        .isLength({ max: 100 })
        .withMessage('Other school type cannot exceed 100 characters'),
    
    body('yearOfJoiningSchool')
        .optional()
        .isInt({ min: 1900, max: new Date().getFullYear() })
        .withMessage('Year of joining school must be between 1900 and current year'),
    
    body('section')
        .optional()
        .trim()
        .isLength({ max: 50 })
        .withMessage('Section cannot exceed 50 characters'),
    
    (req, res, next) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                success: false,
                message: 'Validation failed',
                errors: errors.array()
            });
        }
        next();
    }
];

module.exports = {
    validateRegistration,
    validateLogin,
    validateOTP,
    validatePersonalDetails,
    validatePersonalDetailsUpdate,
    validateSchoolDetails,
    validateSchoolDetailsUpdate,
    validateSchoolDetailsWithoutCurrentClass,
    validateSchoolDetailsWithoutCurrentClassUpdate,
    validateGraduateDetails,
    validateGraduateDetailsUpdate,
    validateSchoolAddition,
    validateCollegeAddition,
    validateWorkingProfessionalDetails,
    validateJobEntry,
    validateBusinessDetails,
    validatePostCreation,
    validatePostUpdate,
    validateComment
};
