# Working Professional & Business Details API

## 🎯 **Overview**
This documentation covers the new API endpoints for managing Working Professional and Business details in the User model.

## 🔧 **User Model Updates**

### **New Objects Added to User Model:**

#### **1. Working Professional Object**
```javascript
workingProfessional: {
    jobCategory: String,        // e.g., "Technology", "Healthcare", "Finance"
    jobSubcategory: String,     // e.g., "Software Development", "Nursing", "Banking"
    state: String,             // e.g., "Karnataka", "Maharashtra"
    district: String,          // e.g., "Bangalore", "Mumbai"
    village: String            // e.g., "Koramangala", "Andheri"
}
```

#### **2. Business Object**
```javascript
business: {
    businessName: String,           // e.g., "ABC Technologies Pvt Ltd"
    businessArea: String,           // e.g., "Software Development", "Manufacturing"
    pincode: String,                // e.g., "560001"
    businessCategory: String,       // e.g., "Technology", "Retail", "Manufacturing"
    businessSubCategory: String,    // e.g., "Software Services", "E-commerce", "Textiles"
    state: String,                  // e.g., "Karnataka", "Gujarat"
    village: String,                // e.g., "Whitefield", "Ahmedabad"
    district: String                // e.g., "Bangalore", "Ahmedabad"
}
```

---

## 📡 **API Endpoints**

### **1. Update Working Professional Details**

#### **Endpoint:**
```bash
PUT /api/users/working-professional/:userId
Authorization: Bearer <token>
Content-Type: application/json
```

#### **Request Body:**
```json
{
    "jobCategory": "Technology",
    "jobSubcategory": "Software Development",
    "state": "Karnataka",
    "district": "Bangalore",
    "village": "Koramangala"
}
```

#### **Response:**
```json
{
    "success": true,
    "message": "Working professional details updated successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "occupation": "working professional",
        "workingProfessional": {
            "jobCategory": "Technology",
            "jobSubcategory": "Software Development",
            "state": "Karnataka",
            "district": "Bangalore",
            "village": "Koramangala"
        }
    }
}
```

#### **Validation:**
- All fields are **required**
- Maximum length validation for each field
- Field trimming applied automatically

---

### **2. Update Business Details**

#### **Endpoint:**
```bash
PUT /api/users/business/:userId
Authorization: Bearer <token>
Content-Type: application/json
```

#### **Request Body:**
```json
{
    "businessName": "ABC Technologies Pvt Ltd",
    "businessArea": "Software Development",
    "pincode": "560001",
    "businessCategory": "Technology",
    "businessSubCategory": "Software Services",
    "state": "Karnataka",
    "village": "Whitefield",
    "district": "Bangalore"
}
```

#### **Response:**
```json
{
    "success": true,
    "message": "Business details updated successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "occupation": "business",
        "business": {
            "businessName": "ABC Technologies Pvt Ltd",
            "businessArea": "Software Development",
            "pincode": "560001",
            "businessCategory": "Technology",
            "businessSubCategory": "Software Services",
            "state": "Karnataka",
            "village": "Whitefield",
            "district": "Bangalore"
        }
    }
}
```

#### **Validation:**
- All fields are **required**
- Maximum length validation for each field
- Field trimming applied automatically

---

### **3. Get Complete Profile Details (Updated)**

The existing profile details endpoint now includes the new objects:

#### **Endpoint:**
```bash
GET /api/users/profile-details/:userId
Authorization: Bearer <token>
```

#### **Response (Updated):**
```json
{
    "success": true,
    "message": "Complete user profile details retrieved successfully",
    "data": {
        // ... existing fields ...
        
        // Working Professional Details
        "workingProfessional": {
            "jobCategory": "Technology",
            "jobSubcategory": "Software Development",
            "state": "Karnataka",
            "district": "Bangalore",
            "village": "Koramangala"
        },
        
        // Business Details
        "business": {
            "businessName": "ABC Technologies Pvt Ltd",
            "businessArea": "Software Development",
            "pincode": "560001",
            "businessCategory": "Technology",
            "businessSubCategory": "Software Services",
            "state": "Karnataka",
            "village": "Whitefield",
            "district": "Bangalore"
        },
        
        // ... other fields ...
        
        // Updated Completion Summary
        "completionSummary": {
            "hasOccupation": true,
            "occupation": "working professional",
            "hasPersonalDetails": true,
            "personalDetailsCompleted": true,
            "occupationCompleted": true,
            "hasSchoolDetails": true,
            "hasCollegeDetails": true,
            "hasWorkingProfessionalDetails": true,
            "hasBusinessDetails": false,
            "hasAbout": true,
            "interestsCount": 4,
            "hasProfileImage": true,
            "hasLocation": true,
            "hasReferralInfo": true
        }
    }
}
```

---

## 🐛 **Location Matching Fix**

### **Issue Fixed:**
The suggested followers API was showing `matchReasons: [ 'Same location: 28' ]` instead of just `'Same location'`.

### **Fix Applied:**
```javascript
// Before
matchReasons.push(`Same location: ${user.state}`);

// After
matchReasons.push(`Same location`);
```

### **Result:**
Now shows: `matchReasons: [ 'Same location' ]` ✅

---

## 📱 **Usage Examples**

### **JavaScript/Node.js**

#### **Update Working Professional Details:**
```javascript
const updateWorkingProfessional = async (token, userId, details) => {
    const response = await fetch(`http://135.181.103.182:5500/api/users/working-professional/${userId}`, {
        method: 'PUT',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(details)
    });
    return await response.json();
};

// Usage
const workingProfessionalDetails = {
    jobCategory: "Technology",
    jobSubcategory: "Software Development",
    state: "Karnataka",
    district: "Bangalore",
    village: "Koramangala"
};

const result = await updateWorkingProfessional('YOUR_TOKEN', 'USER_ID', workingProfessionalDetails);
```

#### **Update Business Details:**
```javascript
const updateBusiness = async (token, userId, details) => {
    const response = await fetch(`http://135.181.103.182:5500/api/users/business/${userId}`, {
        method: 'PUT',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(details)
    });
    return await response.json();
};

// Usage
const businessDetails = {
    businessName: "ABC Technologies Pvt Ltd",
    businessArea: "Software Development",
    pincode: "560001",
    businessCategory: "Technology",
    businessSubCategory: "Software Services",
    state: "Karnataka",
    village: "Whitefield",
    district: "Bangalore"
};

const result = await updateBusiness('YOUR_TOKEN', 'USER_ID', businessDetails);
```

### **cURL Examples**

#### **Update Working Professional:**
```bash
curl -X PUT "http://135.181.103.182:5500/api/users/working-professional/USER_ID" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "jobCategory": "Technology",
    "jobSubcategory": "Software Development",
    "state": "Karnataka",
    "district": "Bangalore",
    "village": "Koramangala"
  }'
```

#### **Update Business:**
```bash
curl -X PUT "http://135.181.103.182:5500/api/users/business/USER_ID" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "businessName": "ABC Technologies Pvt Ltd",
    "businessArea": "Software Development",
    "pincode": "560001",
    "businessCategory": "Technology",
    "businessSubCategory": "Software Services",
    "state": "Karnataka",
    "village": "Whitefield",
    "district": "Bangalore"
  }'
```

### **Flutter/Dart**
```dart
import 'dart:convert';
import 'package:http/http.dart' as http;

class UserDetailsService {
  static const String baseUrl = 'http://135.181.103.182:5500';
  
  static Future<Map<String, dynamic>?> updateWorkingProfessional(
    String token, 
    String userId, 
    Map<String, String> details
  ) async {
    try {
      final response = await http.put(
        Uri.parse('$baseUrl/api/users/working-professional/$userId'),
        headers: {
          'Authorization': 'Bearer $token',
          'Content-Type': 'application/json',
        },
        body: json.encode(details),
      );
      
      if (response.statusCode == 200) {
        final result = json.decode(response.body);
        return result['success'] ? result['data'] : null;
      }
      return null;
    } catch (e) {
      print('Error: $e');
      return null;
    }
  }
  
  static Future<Map<String, dynamic>?> updateBusiness(
    String token, 
    String userId, 
    Map<String, String> details
  ) async {
    try {
      final response = await http.put(
        Uri.parse('$baseUrl/api/users/business/$userId'),
        headers: {
          'Authorization': 'Bearer $token',
          'Content-Type': 'application/json',
        },
        body: json.encode(details),
      );
      
      if (response.statusCode == 200) {
        final result = json.decode(response.body);
        return result['success'] ? result['data'] : null;
      }
      return null;
    } catch (e) {
      print('Error: $e');
      return null;
    }
  }
}

// Usage
final workingProfessionalDetails = {
  'jobCategory': 'Technology',
  'jobSubcategory': 'Software Development',
  'state': 'Karnataka',
  'district': 'Bangalore',
  'village': 'Koramangala'
};

final result = await UserDetailsService.updateWorkingProfessional(
  'YOUR_TOKEN', 
  'USER_ID', 
  workingProfessionalDetails
);
```

---

## 🔍 **Error Handling**

### **Validation Errors:**
```json
{
    "success": false,
    "message": "All fields are required: jobCategory, jobSubcategory, state, district, village"
}
```

### **User Not Found:**
```json
{
    "success": false,
    "message": "User not found"
}
```

### **Server Errors:**
```json
{
    "success": false,
    "message": "Failed to update working professional details",
    "error": "Database connection error"
}
```

---

## ✅ **Features**

### **Working Professional Details:**
- ✅ **Job Category & Subcategory**: Professional categorization
- ✅ **Location Details**: State, district, village
- ✅ **Validation**: Required fields and length limits
- ✅ **Auto-trimming**: Automatic whitespace removal
- ✅ **Debug Logging**: Comprehensive request/response logging

### **Business Details:**
- ✅ **Business Information**: Name, area, category, subcategory
- ✅ **Location Details**: State, district, village, pincode
- ✅ **Validation**: Required fields and length limits
- ✅ **Auto-trimming**: Automatic whitespace removal
- ✅ **Debug Logging**: Comprehensive request/response logging

### **Integration:**
- ✅ **Profile Details API**: Included in complete profile response
- ✅ **Completion Summary**: Added flags for new objects
- ✅ **Authentication**: Protected endpoints with JWT
- ✅ **Error Handling**: Comprehensive error responses

### **Location Matching Fix:**
- ✅ **Clean Display**: Removed state codes from match reasons
- ✅ **Consistent Format**: Shows "Same location" instead of "Same location: 28"

---

## 🎯 **Summary**

The User model now supports:

1. **Working Professional Object**: For users with occupation "working professional"
   - jobCategory, jobSubcategory, state, district, village

2. **Business Object**: For users with occupation "business"
   - businessName, businessArea, pincode, businessCategory, businessSubCategory, state, village, district

3. **API Endpoints**: 
   - `PUT /api/users/working-professional/:userId`
   - `PUT /api/users/business/:userId`

4. **Updated Profile API**: Includes both new objects in complete profile response

5. **Location Matching Fix**: Clean display without state codes

Perfect for managing detailed professional and business information! 🚀
