# Referral System API Documentation

## Overview
The referral system allows users to refer others using unique 6-character codes. Each user gets a unique referral code and can refer others using their code.

## User Model Changes

### New Fields Added:
```javascript
// Referral System
referralCode: {
    type: String,
    unique: true,
    sparse: true, // Allows multiple null values
    uppercase: true,
    trim: true,
    maxlength: [6, 'Referral code must be exactly 6 characters'],
    minlength: [6, 'Referral code must be exactly 6 characters'],
    match: [/^[A-Z0-9]{6}$/, 'Referral code must be exactly 6 uppercase letters and numbers']
},
referredBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    default: null
}
```

## Registration API

### Endpoint
```
POST /api/users/register
```

### Request Body (with referral code)
```json
{
    "name": "John Doe",
    "mobile": "9876543210",
    "password": "password123",
    "email": "john@example.com",
    "dob": "1995-06-15",
    "gender": "Male",
    "emailId": "john.doe@example.com",
    "bloodGroup": "O+",
    "pincodeNative": "110001",
    "state": "28",
    "district": "New Delhi",
    "nativePlace": "Delhi",
    "nameMeaning": "God is gracious",
    "referralCode": "ABC123"
}
```

### Request Body (without referral code - optional)
```json
{
    "name": "Jane Smith",
    "mobile": "9876543211",
    "password": "password123",
    "email": "jane@example.com",
    "dob": "1998-03-20",
    "gender": "Female",
    "emailId": "jane.smith@example.com",
    "bloodGroup": "A+",
    "pincodeNative": "400001",
    "state": "27",
    "district": "Mumbai",
    "nativePlace": "Mumbai",
    "nameMeaning": "God's gracious gift"
}
```

### Response (Success)
```json
{
    "success": true,
    "message": "User registered successfully. Please verify OTP.",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "mobile": "9876543210",
        "referralCode": "XYZ789",
        "referredBy": "68d26d2c0af08456538bc0f5",
        "otpExpires": "2024-01-15T10:15:00.000Z"
    }
}
```

### Response (Success - No Referral Code)
```json
{
    "success": true,
    "message": "User registered successfully. Please verify OTP.",
    "data": {
        "userId": "68d26d2c0af08456538bc0f7",
        "mobile": "9876543211",
        "referralCode": "DEF456",
        "referredBy": null,
        "otpExpires": "2024-01-15T10:15:00.000Z"
    }
}
```

### Response (Error - Invalid Referral Code)
```json
{
    "success": false,
    "message": "Invalid referral code. Please check and try again."
}
```

### Response (Error - Invalid Referral Code Format)
```json
{
    "success": false,
    "message": "Invalid referral code format. Must be 6 uppercase letters and numbers."
}
```

### Response (Error - Self Referral)
```json
{
    "success": false,
    "message": "You cannot use your own referral code."
}
```

## Referral Code Features

### 1. **Unique Generation**
- Each user gets a unique 6-character code
- Format: Uppercase letters and numbers (A-Z, 0-9)
- Examples: `ABC123`, `XYZ789`, `DEF456`

### 2. **Validation**
- Must be exactly 6 characters
- Only uppercase letters and numbers
- Must exist in database
- Cannot be self-referral

### 3. **Optional Field**
- `referralCode` is optional in registration
- If provided, must be valid
- If not provided, `referredBy` will be `null`

### 4. **Database Relationships**
- `referralCode`: User's own unique code
- `referredBy`: ObjectId of the user who referred them
- Both fields are indexed for performance

## Usage Examples

### JavaScript/Node.js
```javascript
// Register with referral code
const registerWithReferral = async () => {
    const response = await fetch('http://135.181.103.182:5500/api/users/register', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            name: "John Doe",
            mobile: "9876543210",
            password: "password123",
            referralCode: "ABC123"
        })
    });
    
    const result = await response.json();
    console.log('Registration result:', result);
    
    if (result.success) {
        console.log('Your referral code:', result.data.referralCode);
        console.log('Referred by:', result.data.referredBy);
    }
};
```

### cURL
```bash
# Register with referral code
curl -X POST http://135.181.103.182:5500/api/users/register \
  -H "Content-Type: application/json" \
  -d '{
    "name": "John Doe",
    "mobile": "9876543210",
    "password": "password123",
    "referralCode": "ABC123"
  }'

# Register without referral code
curl -X POST http://135.181.103.182:5500/api/users/register \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Jane Smith",
    "mobile": "9876543211",
    "password": "password123"
  }'
```

## Debug Logging

The API provides comprehensive debug logging:

```
📝 [REGISTER DEBUG] Registration attempt for mobile: 9876543210
📝 [REGISTER DEBUG] Referral Code: ABC123
🔍 [REGISTER DEBUG] Validating referral code: ABC123
✅ [REGISTER DEBUG] Valid referral code found. Referring user: John Doe (68d26d2c0af08456538bc0f5)
🎯 [REGISTER DEBUG] Generated referral code: XYZ789
✅ [REGISTER DEBUG] User created successfully: 68d26d2c0af08456538bc0f6
```

## Error Handling

### Common Error Scenarios:
1. **Invalid Format**: Code not 6 characters or contains invalid characters
2. **Not Found**: Referral code doesn't exist in database
3. **Self Referral**: User trying to use their own code
4. **Generation Error**: Unable to generate unique code (very rare)

### Error Response Format:
```json
{
    "success": false,
    "message": "Error description",
    "error": "Detailed error message (in development)"
}
```

## Benefits

✅ **Unique Codes**: Each user gets a unique 6-character referral code  
✅ **Optional**: Referral code is completely optional  
✅ **Validation**: Comprehensive validation prevents errors  
✅ **Self-Protection**: Prevents users from referring themselves  
✅ **Debugging**: Full logging for troubleshooting  
✅ **Performance**: Indexed fields for fast lookups  
✅ **Scalable**: Can handle millions of users with unique codes  

## Future Enhancements

- Referral rewards system
- Referral analytics
- Referral leaderboards
- Referral code expiration
- Multiple referral codes per user
