# Notifications System API Documentation

## Overview
The notifications system provides real-time notifications for various user activities including follow requests, post interactions, and cuts interactions.

## Notification Types

### **Follow Notifications**
- **`follow_request`**: Someone sent a follow request
- **`follow_accepted`**: Follow request was accepted
- **`follow_declined`**: Follow request was declined

### **Post Notifications**
- **`post_like`**: Someone liked your post
- **`post_comment`**: Someone commented on your post

### **Cuts Notifications**
- **`cuts_like`**: Someone liked your cut
- **`cuts_comment`**: Someone commented on your cut

## API Endpoints

### **Get Notifications**
```
GET /api/notifications
Authorization: Bearer <token>
```

**Query Parameters:**
- `page` (optional): Page number (default: 1)
- `limit` (optional): Items per page (default: 20)
- `type` (optional): Filter by notification type
- `unreadOnly` (optional): Show only unread notifications (true/false)

**Response:**
```json
{
    "success": true,
    "data": [
        {
            "id": "68d26d2c0af08456538bc0f6",
            "type": "follow_request",
            "title": "New Follow Request",
            "message": "John Doe wants to follow you",
            "isRead": false,
            "priority": "high",
            "age": "2m ago",
            "createdAt": "2024-01-15T10:15:00.000Z",
            "fromUser": {
                "id": "68d26d2c0af08456538bc0f7",
                "name": "John Doe",
                "profileImage": "profile_123.jpg"
            },
            "relatedData": {
                "followId": "68d26d2c0af08456538bc0f8"
            }
        }
    ],
    "pagination": {
        "currentPage": 1,
        "totalPages": 5,
        "totalCount": 100,
        "unreadCount": 15,
        "hasMore": true
    }
}
```

### **Get Unread Count**
```
GET /api/notifications/unread-count
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "unreadCount": 15
}
```

### **Mark Notification as Read**
```
PUT /api/notifications/:notificationId/read
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Notification marked as read",
    "notification": {
        "id": "68d26d2c0af08456538bc0f6",
        "type": "follow_request",
        "title": "New Follow Request",
        "message": "John Doe wants to follow you",
        "isRead": true,
        "priority": "high",
        "age": "2m ago",
        "createdAt": "2024-01-15T10:15:00.000Z"
    }
}
```

### **Mark All Notifications as Read**
```
PUT /api/notifications/mark-all-read
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Marked 15 notifications as read",
    "modifiedCount": 15
}
```

### **Delete Notification**
```
DELETE /api/notifications/:notificationId
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Notification deleted successfully"
}
```

### **Clear All Notifications**
```
DELETE /api/notifications/clear-all
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Deleted 100 notifications",
    "deletedCount": 100
}
```

### **Get Notification Statistics**
```
GET /api/notifications/stats
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "stats": {
        "total": 100,
        "unread": 15,
        "read": 85,
        "byType": {
            "follow_request": 10,
            "follow_accepted": 8,
            "follow_declined": 2,
            "post_like": 25,
            "post_comment": 15,
            "cuts_like": 20,
            "cuts_comment": 20
        }
    }
}
```

## Notification Model

### **Schema Fields**
```javascript
{
    userId: ObjectId,           // User receiving the notification
    fromUserId: ObjectId,       // User who triggered the notification
    type: String,               // Notification type (enum)
    title: String,              // Notification title
    message: String,            // Notification message
    isRead: Boolean,            // Read status
    priority: String,           // Priority level (low/medium/high)
    relatedData: {              // Related data
        postId: ObjectId,       // For post notifications
        cutId: ObjectId,        // For cut notifications
        followId: ObjectId,     // For follow notifications
        commentId: ObjectId     // For comment notifications
    },
    metadata: {                 // Additional metadata
        followStatus: String,   // For follow notifications
        actionCount: Number,    // For aggregated notifications
        platform: String       // Platform info
    },
    createdAt: Date,            // Creation timestamp
    updatedAt: Date             // Last update timestamp
}
```

## Automatic Notification Creation

### **Follow Request Notification**
When a user sends a follow request:
```javascript
await Notification.createFollowRequestNotification(
    followerId,    // User sending the request
    followingId,   // User receiving the request
    followId       // Follow record ID
);
```

### **Follow Accepted Notification**
When a follow request is accepted:
```javascript
await Notification.createFollowAcceptedNotification(
    followerId,    // User who sent the request
    followingId,   // User who accepted
    followId       // Follow record ID
);
```

### **Follow Declined Notification**
When a follow request is declined:
```javascript
await Notification.createFollowDeclinedNotification(
    followerId,    // User who sent the request
    followingId,   // User who declined
    followId       // Follow record ID
);
```

### **Post Like Notification**
When someone likes a post:
```javascript
await Notification.createPostLikeNotification(
    userId,        // Post owner
    postId,        // Post ID
    likerId        // User who liked
);
```

### **Post Comment Notification**
When someone comments on a post:
```javascript
await Notification.createPostCommentNotification(
    userId,        // Post owner
    postId,        // Post ID
    commenterId,   // User who commented
    commentId      // Comment ID
);
```

### **Cuts Like Notification**
When someone likes a cut:
```javascript
await Notification.createCutsLikeNotification(
    userId,        // Cut owner
    cutId,         // Cut ID
    likerId        // User who liked
);
```

### **Cuts Comment Notification**
When someone comments on a cut:
```javascript
await Notification.createCutsCommentNotification(
    userId,        // Cut owner
    cutId,         // Cut ID
    commenterId,   // User who commented
    commentId      // Comment ID
);
```

## Integration Points

### **Follower System Integration**
- **Follow Request**: Automatically creates `follow_request` notification
- **Follow Accepted**: Automatically creates `follow_accepted` notification
- **Follow Declined**: Automatically creates `follow_declined` notification

### **Post System Integration** (To be implemented)
- **Post Like**: Create `post_like` notification
- **Post Comment**: Create `post_comment` notification

### **Cuts System Integration** (To be implemented)
- **Cuts Like**: Create `cuts_like` notification
- **Cuts Comment**: Create `cuts_comment` notification

## Usage Examples

### **JavaScript/Node.js**
```javascript
// Get notifications
const response = await fetch('http://135.181.103.182:5500/api/notifications', {
    headers: {
        'Authorization': 'Bearer ' + token
    }
});
const data = await response.json();

// Mark notification as read
await fetch('http://135.181.103.182:5500/api/notifications/123/read', {
    method: 'PUT',
    headers: {
        'Authorization': 'Bearer ' + token
    }
});

// Get unread count
const unreadResponse = await fetch('http://135.181.103.182:5500/api/notifications/unread-count', {
    headers: {
        'Authorization': 'Bearer ' + token
    }
});
const unreadData = await unreadResponse.json();
console.log('Unread notifications:', unreadData.unreadCount);
```

### **cURL**
```bash
# Get notifications
curl -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/notifications"

# Get unread count
curl -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/notifications/unread-count"

# Mark as read
curl -X PUT \
     -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/notifications/123/read"

# Mark all as read
curl -X PUT \
     -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/notifications/mark-all-read"
```

## Database Indexes

### **Performance Indexes**
```javascript
// Primary query index
{ userId: 1, isRead: 1, createdAt: -1 }

// Type filtering index
{ userId: 1, type: 1, createdAt: -1 }

// From user index
{ fromUserId: 1, type: 1 }

// Time-based index
{ createdAt: -1 }
```

## Features

✅ **Real-time Notifications**: Automatic creation for all user activities  
✅ **Multiple Types**: Support for follow, post, and cuts notifications  
✅ **Read Status**: Track read/unread status  
✅ **Pagination**: Efficient pagination for large notification lists  
✅ **Filtering**: Filter by type and read status  
✅ **Statistics**: Comprehensive notification statistics  
✅ **Bulk Operations**: Mark all as read, clear all notifications  
✅ **Priority Levels**: High, medium, low priority notifications  
✅ **Rich Data**: Include related user and content information  
✅ **Performance Optimized**: Proper indexing for fast queries  

## Future Enhancements

- **Push Notifications**: Mobile push notification support
- **Email Notifications**: Email notifications for important events
- **Notification Preferences**: User-configurable notification settings
- **Aggregated Notifications**: Group similar notifications together
- **Notification Templates**: Customizable notification messages
- **Real-time Updates**: WebSocket integration for live notifications
