# Cuts by User ID API Documentation

## Endpoint
```
GET /api/cuts/user/:userId
```

## Description
Retrieves all cuts created by a specific user with pagination support.

## Parameters

### Path Parameters
- `userId` (required): The ID of the user whose cuts you want to retrieve
  - Type: String
  - Format: MongoDB ObjectId (24 character hex string)
  - Example: `68c7ef413c86477b3b626172`

### Query Parameters
- `page` (optional): Page number for pagination
  - Type: Number
  - Default: 1
  - Example: `?page=2`

- `limit` (optional): Number of cuts per page
  - Type: Number
  - Default: 10
  - Example: `?limit=5`

## Authentication
- **Required**: No authentication required
- **Public Route**: This endpoint is publicly accessible

## Request Examples

### Basic Request
```bash
GET /api/cuts/user/68c7ef413c86477b3b626172
```

### With Pagination
```bash
GET /api/cuts/user/68c7ef413c86477b3b626172?page=1&limit=5
```

### JavaScript/Axios
```javascript
const response = await axios.get('/api/cuts/user/68c7ef413c86477b3b626172?page=1&limit=10');
```

### Flutter/Dart
```dart
final response = await http.get(
  Uri.parse('$baseUrl/api/cuts/user/68c7ef413c86477b3b626172?page=1&limit=10'),
  headers: {
    'Content-Type': 'application/json',
  },
);
```

## Response Format

### Success Response (200)
```json
{
  "success": true,
  "message": "User cuts retrieved successfully",
  "data": {
    "cuts": [
      {
        "_id": "68d2a1ec52d4291170859da5",
        "title": "My Amazing Video Cut",
        "description": "This is a test video cut",
        "visibility": "public",
        "interests": ["Technology", "Entertainment"],
        "tags": ["test", "demo"],
        "video": {
          "fileName": "cut_1758563282411_abc123.mp4",
          "url": "/uploads/cuts/cut_1758563282411_abc123.mp4",
          "originalName": "my_video.mp4",
          "size": 1048576,
          "mimeType": "video/mp4",
          "thumbnail": "/uploads/cuts/cut_1758563282411_abc123_thumb.jpg"
        },
        "user_id": {
          "_id": "68c7ef413c86477b3b626172",
          "name": "John Doe",
          "mobile": "9515446080",
          "profileImage": "profile.jpg"
        },
        "engagement": {
          "likesCount": 5,
          "commentsCount": 3,
          "sharesCount": 2
        },
        "views": 25,
        "isActive": true,
        "createdAt": "2025-01-22T14:00:18.204Z",
        "updatedAt": "2025-01-22T14:00:18.204Z"
      }
    ],
    "pagination": {
      "currentPage": 1,
      "totalPages": 3,
      "totalCuts": 25,
      "hasNextPage": true,
      "hasPrevPage": false
    }
  }
}
```

### Empty Response (No Cuts Found)
```json
{
  "success": true,
  "message": "User cuts retrieved successfully",
  "data": {
    "cuts": [],
    "pagination": {
      "currentPage": 1,
      "totalPages": 0,
      "totalCuts": 0,
      "hasNextPage": false,
      "hasPrevPage": false
    }
  }
}
```

### Error Responses

#### Invalid User ID (400)
```json
{
  "success": false,
  "message": "Invalid user ID format"
}
```

#### User Not Found (404)
```json
{
  "success": false,
  "message": "User not found"
}
```

#### Server Error (500)
```json
{
  "success": false,
  "message": "Failed to get user cuts",
  "error": "Error message details"
}
```

## Response Fields

### Cut Object
- `_id`: Unique identifier for the cut
- `title`: Title of the cut
- `description`: Description of the cut
- `visibility`: Visibility setting (public, private, friends, group)
- `interests`: Array of interest tags
- `tags`: Array of custom tags
- `video`: Video file information
  - `fileName`: Generated filename
  - `url`: Public URL to access the video
  - `originalName`: Original filename
  - `size`: File size in bytes
  - `mimeType`: MIME type of the video
  - `thumbnail`: URL to video thumbnail
- `user_id`: User information (populated)
  - `_id`: User ID
  - `name`: User's name
  - `mobile`: User's mobile number
  - `profileImage`: User's profile image URL
- `engagement`: Engagement metrics
  - `likesCount`: Number of likes
  - `commentsCount`: Number of comments
  - `sharesCount`: Number of shares
- `views`: Total view count
- `isActive`: Whether the cut is active
- `createdAt`: Creation timestamp
- `updatedAt`: Last update timestamp

### Pagination Object
- `currentPage`: Current page number
- `totalPages`: Total number of pages
- `totalCuts`: Total number of cuts for the user
- `hasNextPage`: Whether there's a next page
- `hasPrevPage`: Whether there's a previous page

## Features

### ✅ Implemented Features
- **Pagination Support**: Efficient pagination with page and limit parameters
- **User Population**: Automatically populates user details for each cut
- **Sorting**: Cuts are sorted by creation date (newest first)
- **Active Filter**: Only returns active cuts (isActive: true)
- **Error Handling**: Comprehensive error handling for invalid inputs
- **Performance**: Optimized database queries with skip and limit

### 🔍 Query Behavior
- **Default Limit**: 10 cuts per page if not specified
- **Default Page**: Page 1 if not specified
- **Sorting**: Always sorted by `createdAt` in descending order (newest first)
- **Filtering**: Only returns cuts where `isActive: true`

## Usage Examples

### Get User's Latest Cuts
```bash
GET /api/cuts/user/68c7ef413c86477b3b626172
```

### Get User's Cuts with Custom Pagination
```bash
GET /api/cuts/user/68c7ef413c86477b3b626172?page=2&limit=5
```

### Get All User's Cuts (Large Limit)
```bash
GET /api/cuts/user/68c7ef413c86477b3b626172?page=1&limit=100
```

## Testing

Run the test script to verify the endpoint:
```bash
node scripts/testCutsByUserIdAPI.js
```

## Related Endpoints

- `GET /api/cuts` - Get all cuts with filtering
- `GET /api/cuts/trending` - Get trending cuts
- `GET /api/cuts/interests/:interests` - Get cuts by interests
- `GET /api/cuts/:cutId` - Get specific cut by ID
- `POST /api/cuts` - Create new cut
- `PUT /api/cuts/:cutId` - Update cut
- `DELETE /api/cuts/:cutId` - Delete cut

## Notes

- This endpoint is publicly accessible and does not require authentication
- Only active cuts are returned (isActive: true)
- User details are automatically populated for each cut
- Cuts are always sorted by creation date (newest first)
- Empty arrays are returned for users with no cuts (not an error)
